/*
 * sirf_osp.c
 * See http://dev.gateworks.com/w2sg0008i/One_Socket_Protocol_Interface_Control_Document_(OSP_ICD)_(CS-129291-DC-15).pdf
 *
 * usage: sirf_osp /dev/ttymxc4 <current baudrate> [command [arguments...]]
 *		*Note that GW51XX boards use /dev/ttymxc0
 *
 * Commands:
 *   osp <baudrate> - switch from NMEA to OSP
 *   lna <low|high> - set LNA mode low or high
 *   config - read config (shows LNA setting in response)
 *   nmea <baudrate> - switch OSP to NMEA (GSV/GGA msgs 1Hz no csums)
 *
 * will continuously read and display packets
 *
 * The following is the list of valid bauds - 900, 1200, 1800, 2400,
 * 3600, 4800, 7200, 9600, 14400, 19200, 28800, 38400, 57600, 76800, 115200,
 * 153600, 230400, 307200, 460800, 614400, 921600, 1228800, and 1843200.
 *
 * Gateworks Location: 35.258175, -120.668927 (North, East)
 * Negative location values represent opposite compass directions
 *
 * p.332 - table of geodetic navigation data (holds fix information)
 * p.607 - descriptions of byte type values in message description tables
 */

#include <ctype.h>
#include <errno.h>
#include <stdlib.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <termios.h>
#include <time.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/serial.h>
#include <math.h>

//Actual location * 10^7
#define REAL_LAT 352581750
#define REAL_LONG -1206689270
#define DEBUG

struct tracker_cfg {
        uint8_t mid;
        uint8_t sid;
        uint32_t refclk_freq;
        uint16_t refclk_delay;
        uint32_t refclk_uncert;
        int32_t refclk_drift;
        uint8_t lna;
        uint8_t pincfg;
        uint8_t iocfg[22];
        uint8_t uart_preamble;
        uint8_t uart_delay;
        uint32_t uart_baud;
        uint8_t uart_flowctrl;
        uint16_t i2c_master;
        uint16_t i2c_slave;
        uint8_t i2c_rate;
        uint8_t i2c_mode;
        uint16_t i2c_len;
        uint8_t power_control;
        uint8_t power_cfg;
} __attribute__ ((__packed__));

struct init_data_source {
	uint8_t mid;
	int32_t ecef_x;
	int32_t ecef_y;
	int32_t ecef_z;
	int32_t clk_drif;
	uint32_t time_of_week;
	uint16_t week;
	uint8_t ch;
	uint8_t reset_cfg;
} __attribute__ ((__packed__));

struct nmea {
	uint8_t mid;
	uint8_t mode;
	uint8_t gga_period;
	uint8_t gga_csum;
	uint8_t gll_period;
	uint8_t gll_csum;
	uint8_t gsa_period;
	uint8_t gsa_csum;
	uint8_t gsv_period;
	uint8_t gsv_csum;
	uint8_t rmc_period;
	uint8_t rmc_csum;
	uint8_t vtg_period;
	uint8_t vtg_csum;
	uint8_t mss_period;
	uint8_t mss_csum;
	uint8_t epe_period;
	uint8_t epe_csum;
	uint8_t zda_period;
	uint8_t zda_csum;
	uint8_t resv[2];
	uint16_t bitrate;
} __attribute__ ((__packed__));

struct config {
	uint8_t mid;
	uint8_t sid;
	uint32_t baud;
	uint32_t i2c_host;
	uint32_t i2c_4e;
	uint32_t wait_sates;
	uint32_t tcxo_warmup;
	uint32_t cgee_dis;
	uint32_t max_epe;
	uint16_t gpio0;
	uint16_t gpio1;
	uint16_t gpio2;
	uint16_t gpio3;
	uint16_t gpio4;
	uint16_t gpio5;
	uint16_t gpio6;
	uint16_t gpio7;
	uint16_t rx;
	uint16_t tx;
	uint16_t max_att;
	uint8_t mems_i2c_addr0;
	uint8_t mems_i2c_addr1;
	uint8_t mems_i2c_addr2;
	uint8_t mems_i2c_addr3;
} __attribute__ ((__packed__));

enum err {
	OSP_OK = 0,
	OSP_READERR,
	OSP_OVERFLOW,
	OSP_CRCERR,
};

static void hexdump(unsigned char *buf, int size)
{
	int i = 0;
	char ascii[20];

	ascii[0] = 0;
	for (i = 0; i < size; i++) {
		if (0 == (i % 16)) {
			if (ascii[0]) {
				ascii[16] = 0;
				printf("  |%s|\n", ascii);
				ascii[0] = 0;
			}
			printf("%08x ", i);
		}
		if (0 == (i % 8))
			printf(" ");
		printf("%02x ", buf[i]);
		ascii[i % 16] = (buf[i] < ' ' || buf[i] > 127) ? '.' : buf[i];
		ascii[(i % 16)+1] = 0;
	}
	for (; i % 16; i++) {
		if (0 == (i % 8))
			printf(" ");
		printf("   ");
	}
	printf("  |%-16s|\n", ascii);
}

/* Tracks estimated moving averages of different alphas
 * that represent the distance (in degrees) between
 * actual location and the device measured location
 */
static double* ema_update(int32_t latit, int32_t longit)
{
	int i;
	static double alpha[4] = {0, .1, .05, .01}, ema[4] = {0};
	double distance;

	// If the fix has not been acquired, ignore
	if (!latit && !longit && !ema[0]) return ema;

	// Calculate distance from real position (in degrees)
	distance = sqrt(pow(latit - REAL_LAT, 2) + pow(longit - REAL_LONG, 2));
	// Reverse the scaling applied by gps module
	distance /= pow(10, 7);

	// fix acquired but emas not initialized
	// Use first value as init flag
	if (!ema[0] && (ema[0] = 1))
		for(i=1; i <= 3; ++i)
			ema[i] = distance;
	else
		for(i=1; i <= 3; ++i)
			ema[i] = ema[i] * (1 - alpha[i]) + distance * alpha[i];

	return ema;
}

#define BYTE(x) buf[x]
#define WORD(x) buf[x]<<8|buf[x+1]
#define DWORD(x) buf[x]<<24|buf[x+1]<<16|buf[x+2]<<8|buf[x+3]
#define TYPE(fmt, args...) sprintf(msg_short+strlen(msg_short), fmt, args)
#define DESC(fmt, args...) sprintf(msg_long+strlen(msg_long), fmt, args)

int print_osp(const char* prefix, uint8_t *buf, int size, int csvmode)
{
	int i,j,k,l;
	char msg_short[64];
	char msg_long[1024];
	const char *name;
	int32_t latit = 0, longit = 0;
	static long msgnum = 0;
	double *ema;

	msg_short[0] = 0;
	msg_long[0] = 0;
	switch(buf[0]) {
		case 2:
			name = "Measure Navidation Data Out";
			break;
		case 4: {
			int sats = 0;
			int sig = 0;

			name = "Measured Tracker Data Out";
			DESC("\tweek:0x%04x\n", WORD(1));
			DESC("\tTOW:0x%08x\n", DWORD(3));
			DESC("\tchans: (%d)\n", BYTE(7));
			l = 0;
			for (i = 0; i < 12; i++) {
				if ((WORD(11+(i*15))) == 0)
					continue;
				sats++;
				DESC("\t\t%02d:%03d:%03d:0x%04x",
						BYTE(8+(i*15)), // SVid
						BYTE(9+(i*15)), // Azimuth
						BYTE(10+(i*15)), // Elev
						WORD(11+(i*15)) // State
				   );
				/* http://www.insidegnss.com/auto/novdec10-Solutions.pdf:
				 *
				 * SNR is usually expressed in terms of decibels and refers to the ratio of
				 * the signal power and noise power in a given bandwidth.
				 *
				 * C/N0 on the other hand is usually expressed in decibel-Hertz and refers
				 * to the ratio of the carrier poewr and the noise power per unit bandwidth.
				 * Typical  values in an L1 C/A code receiver is 37 to 45dbHz.
				 */
				for (j = 0, k = 0; k < 10; k++) {
					j += BYTE(13+(i*15)+k);
					//DESC("%02d ", BYTE(13+(i*15)+k));
				}
				DESC(": %02d\n", j/10);
				sig += (j/10);
			}
			TYPE("Sats:%d/12 Avg:%d", sats, sats?sig/sats:0);
		}	break;
		case 6:
			name = "Software Version String";
			TYPE("Reply to MID %d", 132);
			DESC("Sirf Version: %s\nCustomer Version: %s\n"
				, buf+3, buf+4 + strlen(buf+3));
			break;
		case 9:
			name = "CPU Throughput";
			break;
		case 11:
			name = "Command Acknowledgement";
			TYPE("Ack:%d ID:0x%02x", BYTE(1), BYTE(2));
			break;
		case 18:
			name = "OkToSend";
			TYPE(":%s", BYTE(1)?"NO":"YES");
			break;
		case 41:
			name = "Geodetic Navigation Data";
			//hexdump(buf, size);
			DESC("\tnav valid:0x%04x\n", WORD(1));
			DESC("\tnav type :0x%04x\n", WORD(3));
			DESC("\tweek     :0x%04x\n", WORD(5));
			DESC("\tTOW      :0x%08x\n", DWORD(7));
			DESC("\tUTC      :%04d%02d%02d %02d:%02d:%02d\n\tsats:",
					WORD(11), BYTE(13), BYTE(14),
					BYTE(15), BYTE(16), WORD(17));
			for(i=0,j=0; i < 32; i++) {
				if (1<<i & (DWORD(19))) {
					DESC(" %02d", i + 1);
					j++;
				}
			}
			latit = (int32_t)DWORD(23);
			longit = (int32_t)DWORD(27);
			DESC("\n\tFIX: %ldN %ldE\n", latit, longit);
			ema = ema_update(latit, longit);
			DESC("\tDegree Difference EMA's (.1/.05/.01): %f/%f/%f\n", ema[1], ema[2], ema[3]);
			TYPE("Sats used in fix:%d", j);
			break;
		case 51: name = "Tracker Load Status Report"; break;
		case 56:
			switch(buf[1]) {
			case 41: name = "SIF Aiding Status"; break;
			case 42: name = "SIF Status Message"; break;
			} break;
		/*
		case 56: // Extended Ephemeris Data/SGEE Download Output
		switch(buf[1]) {
		case 0x01: name = "GPS Data and Ephemeris Mask"; break;
		case 0x02: name = "Extended Ephemeris Integrity"; break;
		case 0x03: name = "Extended Ephemeris Integrity"; break;
		case 0x04: name = "EE Provide Synthesized Ephemereis Clock Bias Adjustment"; break;
		case 0x05: name = "Verified 50bps Broadcast"; break;
		} break;
		*/
		case 65:
			switch(buf[1]) {
			case 192:
				name = "GPIO State Output";
				TYPE("State:0x%02x%02x", buf[2], buf[3]);
				break;
		} break;
		case 71:
			name = "Hardware Configuration Request";
			TYPE("ID:0x%02x", BYTE(2));
			break;
		case 92:
			switch(buf[1]) {
			case 1:
				   name = "CW Interference Report";
				   break;
			case 2:
				   name = "CW Mitigation Report";
				   break;
			} break;
		case 93:
			name = "TCXO Learning TCXO Uncertainty";
			break;
		case 128:
			name = "Initialize Data Source";
			break;
		case 132:
			name = "Poll Software Version";
			break;
		case 178:
			switch(buf[1]) {
			/* SW Toolbox Input Requests */
			case  1: name = "MeiToCustomlo"; break;
			case  2: {
				struct tracker_cfg *cfg = (struct tracker_cfg *)buf;
				name = "Tracker Configuration";
				TYPE("LNA:%d", cfg->lna);
			}	break;
			case  3: name = "PeekPoke"; break;
			case  9: name = "Poll Tracker Configuration"; break;
			case 11: name = "Poll Customer Configuration"; break;
 			/* SW Toolbox Output Response */
			case  4: name = "Tracking Peek/Poke Response"; break;
			case  5: name = "Flash store Response"; break;
			case  6: name = "Flash erase Response"; break;
			case  7: name = "Tracker Configuration Response"; break;
			case  8: name = "MeiToCustomlo REsponse"; break;
			case 10: name = "Tracker Config Poll Response"; break;
			case 12:
				name = "Customer Config Poll Response";
				//hexdump(buf, size);
				TYPE("LNA:%d", BYTE(0x51));
				break;
			} break;
		default:
 			name = "*** Unhandled ***";
			break;
	}
	if (!csvmode) {
		printf("%s: %s MID %d SID %d (%d bytes) %s\n", prefix, name,
				buf[0], buf[1], size, msg_short);
		if (*msg_long)
			printf("%s\n", msg_long);
		if (!*name)
			return -1;
	}
	else {
		if (latit && longit)
			printf("%ld,%lf,%lf\n", ++msgnum, latit/pow(10, 7), longit/pow(10, 7));
	}
	return 0;
}

int read_bytes(int fd, uint8_t *buf, int size)
{
	int i, rz;

	i = 0;
	while (i < size) {
		rz = read(fd, buf + i, size - i);
		if (rz < 0) {
			perror("read");
			return -errno;
		}
		if (rz == 0)
			continue;
		i += rz;
	}

	return i;
}

int read_osp(int fd, uint8_t *buf, int size)
{
	int i, rz, sz;
	uint8_t val;
	uint16_t len, csum;
	uint8_t hdr[2];

	// read until we have a valid start sequence (0xa0 0xa2)
sync:
	rz = read_bytes(fd, hdr, 2);
	if (rz < 0)
		goto err;
	if (hdr[0] != 0xa0 || hdr[1] != 0xa2)
		goto sync;

	// 2-byte length
	rz = read_bytes(fd, hdr, 2);
	if (rz < 0)
		goto err;
	len = hdr[1] | (hdr[0] << 8);

	// payload
	sz = (len > size) ? size : len;
	rz = read_bytes(fd, buf, sz);
	if (rz < 0)
		goto err;

	// checksum
	csum = 0;
	for (i = 0; i < sz;i++ )
		csum += buf[i];
	rz = read_bytes(fd, hdr, 2);
	if (rz < 0)
		goto err;
	if (csum != (hdr[1] | (hdr[0] << 8))) {
		fprintf(stderr, "failed csum\n");
		goto sync;
	}

	// end sequence
	rz = read_bytes(fd, hdr, 2);
	if (rz < 0)
		goto err;
	if (hdr[0] != 0xb0 || hdr[1] != 0xb3) {
		fprintf(stderr, "failed end seq\n");
		goto sync;
	}

	return len;

err:
	fprintf(stderr, "read failed: %d\n", rz);
	return errno;
}

// Use a sid argument of -1 if MID has no SID
int wait_osp(int fd, uint8_t mid, uint8_t sid, uint8_t *buf, int size)
{
	int sz, i;

	printf("Waiting for MID %d", mid);

	if (sid != 255)
		printf(" SID %d:\n", sid);
	else
		puts("");

	for(i = 0; ; i++) {
		sz = read_osp(fd, buf, size);
		if (sz < 1)
			break;
		if (buf[0] == mid && (buf[1] == sid || sid == 255))
			break;
		// indent to indicate we received this but are waiting for
		// something else
		print_osp("  Recv (skipped)", buf, sz, 0);
	}
	// show received msg
	print_osp("Recv", buf, sz, 0);

	return sz;
}

int write_osp(int fd, uint8_t *_buf, int size)
{
	int sz, i;
	uint8_t buf[2048];
	uint16_t csum;

#ifdef DEBUG
	printf("%s: %d bytes (%d byte payload):\n", __func__, size+8, size);
#endif
	buf[0] = 0xa0;
	buf[1] = 0xa2;
	buf[2] = size >> 8;
	buf[3] = size & 0xff;
	csum = 0;
	for (i = 0; i < size;i++ ) {
		buf[4+i] = _buf[i];
		csum += _buf[i];
	}
	buf[size+4] = csum >> 8;
	buf[size+5] = csum & 0xff;
	buf[size+6] = 0xb0;
	buf[size+7] = 0xb3;
#ifdef DEBUG
	hexdump(buf, size+8);
#endif
	if (write(fd, buf, size+8) != (size+8))
		goto err;

	return size;
err:
	perror("write failed\n");
	return errno;
}

int send_nmea(int fd, const char *msg)
{
	int i, csum, sz;
	uint8_t buf[1024];

	printf("%s: '%s'\n", __func__, msg);
	csum = 0;
	sz = strlen(msg);
	for (i = 0; i < sz && msg[i]; i++)
		csum ^= msg[i];
	sprintf(buf, "$%s*%02d\r\n", msg, csum);
#ifdef DEBUG
	printf("Sending %d bytes (csum=%d):\n", (int) strlen(buf), csum);
	hexdump(buf, strlen(buf));
#endif
	sz = write(fd, buf, strlen(buf));
	if (sz < 0)
		perror("write failed");
	if (sz < strlen(buf)) {
		printf("write short\n");
		return sz;
	}

	return 0;
}

int set_baud(int fd, int baud)
{
	struct termios ttystate;
	speed_t speed;

	tcgetattr(fd, &ttystate);
	switch(baud) {
		case 1200: speed = B1200; break;
		case 2400: speed = B2400; break;
		case 4800: speed = B4800; break;
		case 9600: speed = B9600; break;
		case 19200: speed = B19200; break;
		case 38400: speed = B38400; break;
		case 57600: speed = B57600; break;
		case 115200: speed = B115200; break;
		case 230400: speed = B230400; break;
		case 460800: speed = B460800; break;
		default:
			fprintf(stderr, "invalid baud rate %d\n", baud);
			return -EINVAL;
			break;
	}
	printf("Setting baudrate to %d\n", baud);
	if (cfsetispeed(&ttystate, speed))
		perror("cfsetispeed");
	if (cfsetospeed(&ttystate, speed))
		perror("cfsetospeed");

	if (tcsetattr(fd, TCSANOW, &ttystate))
		perror("tcsetattr");

	return 0;
}

/** main function
 */
int main(int argc, char** argv)
{
	struct termios orig_ttystate, ttystate;
	int fd, sz, rz, bytes, csvmode = 0;
	const char *baud, *dev;
	int timeout = 2; // time in seconds to wait for response
	char buf[2048];
	time_t start;

	if (argc < 3) {
		fprintf(stderr,
			"usage: %s <device> <baud> [command]\n\n"
			"\tCommands:\n"
			"lna <low|high>  - set LNA mode low or high\n"
			"config          - read config (shows LNA setting in response)\n"
			"version         - show software version of device\n"
			"osp <baudrate>  - switch from NMEA to OSP\n"
			"nmea <baudrate> - switch OSP to NMEA (GSV/GGA msgs 1Hz no csums)\n"
			"csv <outfile>   - write fix data in csv format to \"outfile\"\n\n",
			argv[0]);
		exit(1);
	}

	dev = argv[1];
	baud = argv[2];

	// open device
	fd = open(dev, O_RDWR | O_NONBLOCK | O_NOCTTY);
	if (fd <= 0) {
		perror("open");
		exit(-1);
	}

	// get original ttystate
	tcgetattr(fd, &orig_ttystate);

	// create a sane TTY state (raw mode, no HW/SF flow control)
	memset(&ttystate, 0, sizeof(ttystate));

	// 8 data bits, no parity, 1 stop bit
	// enable receiver and ignore modem status lines
	ttystate.c_cflag = CS8 | CREAD | CLOCAL;

	// set tty state
	if (tcsetattr(fd, TCSANOW, &ttystate))
		perror("tcsetattr");

	// baudrate
	set_baud(fd, atoi(baud));

	// flush in/out data
	tcflush(fd, TCIOFLUSH);

	/* switch from NMEA/4800 to OSP
	 * usage: osp <baudrate>
	 */
	if (argc > 4 && strcasecmp(argv[3], "osp") == 0) {
		int baud = atoi(argv[4]);

		printf("Switching from NMEA to OSP:\n");
		sprintf(buf, "PSRF100,0,%d,8,1,0", baud);
		send_nmea(fd, buf);
	}

	/* Send a Tracker Configuration message - MID 178, SID 2
	 * usage: lna <low|high>
	 */
	else if (argc > 4 && strcasecmp(argv[3], "lna") == 0) {
		struct tracker_cfg *cfg = (struct tracker_cfg *)buf;
		struct init_data_source *init = (struct init_data_source *)buf;
		uint8_t iocfg[] = {
			0x03, 0xfc, // IO1
			0x03, 0xfc, // IO2
			0x00, 0x04, // IO3
			0x00, 0x3e, // IO4
			0x00, 0x00, // IO5
			0x00, 0x7c, // IO6
			0x00, 0x00, // IO7
			0x00, 0x00, // IO8
			0x00, 0x00, // IO9
			0x00, 0x00, // IO10
			0x00, 0x00  // IO11
		};


		/* Send Tracker Configuration message - MID 178, SID 2 */
		memset(buf, 0, sizeof(*cfg));
		cfg->mid = 178;
		cfg->sid = 2;
		cfg->refclk_freq = htonl(16369000);
		cfg->refclk_delay = htons(1023);
		cfg->refclk_uncert = htonl(3000);
		cfg->refclk_drift = htonl(96250);
		if (strcasecmp(argv[4], "low") == 0) {
			printf("setting LNA low\n");
			cfg->lna = 1; // low gain
		}
		else
			printf("setting LNA high\n");
		cfg->pincfg = 1;
		memcpy(cfg->iocfg, iocfg, sizeof(iocfg));
		cfg->uart_preamble = 0;
		cfg->uart_delay = 0;
		cfg->uart_baud = htonl(115200);
		cfg->uart_flowctrl = 0;
		cfg->i2c_master = htons(0x62);
		cfg->i2c_slave = htons(0x60);
		cfg->i2c_rate = 1;
		cfg->i2c_mode = 1;
		cfg->i2c_len = htons(500);
		cfg->power_control = 0x2a;
		cfg->power_cfg = 0x01;
		print_osp("Send", buf, sizeof(*cfg), 0);
		if (write_osp(fd, buf, sizeof(*cfg)) != sizeof(*cfg))
			fprintf(stderr, "Failed setting LNA %s\n", argv[4]);
		/* Wait for Tracking Configuration Response - MID 178, SID 7 */
		wait_osp(fd, 178, 7, buf, sizeof(buf));
		/* Wait for Command Ack */
		wait_osp(fd, 11, 178, buf, sizeof(buf));

		/* Send 'Initialize Data Source - MID 128' */
		memset(buf, 0, sizeof(*init));
		init->mid = 128;
		init->ch = 12;
		init->reset_cfg = 0; /* Table 5.34 */

		/* Send: Initialize Data Source - MID 128 */
		print_osp("Send", buf, sizeof(*init), 0);
		if (write_osp(fd, buf, sizeof(*init)) != sizeof(*init))
			fprintf(stderr, "Failed hot start\n");
		wait_osp(fd, 11, 128, buf, sizeof(buf));
	}

	/* Send a 'Switch to NMEA Protocol - MID 129
	 * usage: nmea <baudrate>
	 */
	else if (argc > 4 && strcasecmp(argv[3], "nmea") == 0) {
		struct nmea *nmea = (struct nmea*)buf;

		memset(buf, 0, sizeof(*nmea));
		nmea->mid = 129;
		nmea->mode = 0;
		nmea->gga_period = 1;
		nmea->gsv_period = 1;
		nmea->bitrate = htons(atoi(argv[4]));
		print_osp("Send", buf, sizeof(*nmea), 0);
		if (write_osp(fd, buf, sizeof(*nmea)) != sizeof(*nmea))
			fprintf(stderr, "Failed nmea\n");
	}

	/* Send a 'Poll Customer Configuration Kit Parameters - MID 178, SID 11
	 * usage: config
	 * A 'Customer Configuration Kit Poll Response - MID 178, SID 12' will
	 * show config.
	 */
	else if (argc > 3 && strcasecmp(argv[3], "config") == 0) {
		buf[0] = 178;
		buf[1] = 11;
		print_osp("Send", buf, 2, 0);
		if (write_osp(fd, buf, 2) != 2)
			fprintf(stderr, "Failed poll params\n");
		wait_osp(fd, 178, 12, buf, sizeof(buf));
	}

	/* Send a 'Poll Software Version – MID 132' p.120
	 * usage: version
	 * A 'Software Version String (Response to Poll) – MID 6' will
	 * return a string with version info. p.282
	 */
	else if (argc > 3 && strcasecmp(argv[3], "version") == 0) {
		buf[0] = 132;
		buf[1] = 0;
		print_osp("Send", buf, 2, 0);
		if (write_osp(fd, buf, 2) != 2)
			fprintf(stderr, "Failed poll params\n");
		wait_osp(fd, 6, -1, buf, sizeof(buf));
	}

	/* endless read/print loop */
	else {
		if (argc > 4 && strcasecmp(argv[3], "csv") == 0) {
			freopen(argv[4], "w", stdout);
			csvmode = 1;
			printf("count,latitude,longitude\n");
		}
		else
			printf("Listening for OSP:\n");
		for(;;) {
			rz = read_osp(fd, buf, sizeof(buf));
			if (rz < 1)
				break;
			if (print_osp("Recv", buf, rz, csvmode) < 0)
				break;
			fflush(stdout);
		}
	}

	// restore terminal state
	tcsetattr(fd, TCSANOW, &orig_ttystate);

	close(fd);

	return 0;
}
